﻿' 版权所有 (C) Microsoft Corporation。保留所有权利。
Option Explicit Off
Option Infer On
Option Strict On

Imports SampleQueries.SampleSupport
Imports System.IO
Imports System.Reflection
Imports System.Linq
Imports System.Xml.Linq


<Title("LINQ 查询示例"), _
Prefix("Linq")> _
Public Class LinqSamples
    Inherits SampleHarness

    Private dataPath As String = System.Windows.Forms.Application.StartupPath & "\..\..\SampleData\"

    Public Class Customer
        Public CustomerID As String
        Public CompanyName As String
        Public Address As String
        Public City As String
        Public Region As String
        Public PostalCode As String
        Public Country As String
        Public Phone As String
        Public Fax As String
        Public Orders As Order()
    End Class

    Public Class Order
        Public OrderID As Integer
        Public OrderDate As DateTime
        Public Total As Decimal
    End Class

    Public Class Product
        Public ProductID As Integer
        Public ProductName As String
        Public Category As String
        Public UnitPrice As Decimal
        Public UnitsInStock As Integer
    End Class

    Private productList As List(Of Product)
    Private customerList As List(Of Customer)


    <Category("您的第一个 LINQ 查询"), _
    Title("筛选数字"), _
    Description("此查询将返回数组中小于 5 的所有数字。")> _
    Public Sub Linq1()

        '创建数字数组
        Dim numbers = New Integer() {5, 4, 1, 3, 9, 8, 6, 7, 2, 0}

        '此查询将给出数组中小于 5 的所有数字
        '变量 num 称为“范围变量”- 它将依次
        '代表数组中的每个值。
        Dim lowNums = From num In numbers _
                      Where num < 5 _
                      Select num

        Console.WriteLine("Numbers < 5:")

        'lowNums 现在仅包含小于 5 的数字
        '可以使用循环输出每个值
        For Each lowNumber In lowNums
            Console.WriteLine(lowNumber)
        Next
    End Sub

    <Category("筛选(Where)")> _
    <Title("Where - 简单用法 1")> _
    <Description("此示例使用 Where 子句查找所有缺货产品。")> _
    Public Sub Linq2()
        Dim products = GetProductList()

        '只需要范围变量时可以不使用 Select 语句
        Dim soldOutProducts = From prod In products _
                              Where prod.UnitsInStock = 0 _
                              Select prod

        Console.WriteLine("Sold out products: ")
        For Each prod In soldOutProducts
            Console.WriteLine(prod.ProductName & " is sold out!")
        Next
    End Sub

    <Category("筛选(Where)")> _
    <Title("Where - 简单用法 2")> _
    <Description("此示例使用 Where 子句查找所有现货且" & _
                 "单价大于 3.00 的产品。")> _
    Public Sub Linq3()
        Dim products = GetProductList()

        Dim expensiveProducts = From prod In products _
                                Where prod.UnitsInStock > 0 AndAlso prod.UnitPrice > 3.0

        Console.WriteLine("In-stock products that cost more than 3.00:")
        For Each prod In expensiveProducts
            Console.WriteLine(prod.ProductName & "is in stock and costs more than 3.00.")
        Next
    End Sub

    <Category("筛选(Where)")> _
    <Title("Where - 深化")> _
    <Description("此示例使用 Where 子句查找所有在华盛顿的客户，" & _
                 "然后使用所得序列深化订单")> _
    Public Sub Linq4()
        Dim customers = GetCustomerList()

        Dim waCustomers = From cust In customers _
                          Where cust.Region = "WA" _
                          Select cust

        Console.WriteLine("Customers from Washington and their orders:")
        For Each cust In waCustomers
            Console.WriteLine("Customer " & cust.CustomerID & ": " & cust.CompanyName)
            For Each ord In cust.Orders
                Console.WriteLine("  Order " & ord.OrderID & ": " & ord.OrderDate)
            Next
        Next
    End Sub

    <Category("筛选(Where)")> _
    <Title("Where - 带索引")> _
    <Description("此示例演示带索引的 Where 子句，该子句返回其名称短于" & _
                 "其值的数字。")> _
    Public Sub Linq5()
        Dim digits = New String() {"zero", "one", "two", "three", "four", "five", "six", "seven", "eight", "nine"}

        Dim shortDigits = digits.Where(Function(digit, index) digit.Length < index)

        Console.WriteLine("Short digits:")
        For Each d In shortDigits
            Console.WriteLine("The word " & d & " is shorter than its value.")
        Next
    End Sub

    <Category("投影(Select)")> _
    <Title("Select - 简单用法 1")> _
    <Description("此示例使用 Select 生成一个整数序列，" & _
                 "其中每个整数都比现有整数数组中的相应整数大一。")> _
    Public Sub Linq6()
        Dim numbers = New Integer() {5, 4, 1, 3, 9, 8, 6, 7, 2, 0}

        Dim numsPlusOne = From num In numbers _
                          Select num + 1

        Console.WriteLine("Numbers + 1:")
        For Each i In numsPlusOne
            Console.WriteLine(i)
        Next
    End Sub

    <Category("投影(Select)")> _
    <Title("Select - 简单用法 2")> _
    <Description("此示例使用 Select 返回仅包含产品列表中产品名称的序列。")> _
    Public Sub Linq7()
        Dim products = GetProductList()

        Dim productNames = From prod In products _
                           Select prod.ProductName

        Console.WriteLine("Product Names:")
        For Each productName In productNames
            Console.WriteLine(productName)
        Next
    End Sub

    <Category("投影(Select)")> _
    <Title("Select - 转换")> _
    <Description("此示例使用 Select 生成表示" & _
                 "整数序列文本形式的字符串序列。")> _
    Public Sub Linq8()
        Dim numbers = New Integer() {5, 4, 1, 3, 9, 8, 6, 7, 2, 0}
        Dim stringNames = New String() {"zero", "one", "two", "three", "four", "five", "six", "seven", "eight", "nine"}

        Dim textNums = From num In numbers _
                       Select stringNames(num)

        Console.WriteLine("Number strings:")
        For Each s In textNums
            Console.WriteLine(s)
        Next
    End Sub

    <Category("投影(Select)")> _
    <Title("Select - 匿名类型 1")> _
    <Description("此示例使用 Select 生成" & _
                 "原始数组中每个单词的大写和小写形式的序列。")> _
    Public Sub Linq9()
        Dim words = New String() {"aPPLE", "BlUeBeRrY", "cHeRry"}

        Dim upperLowerWords = From word In words _
                              Select Upper = word.ToUpper(), Lower = word.ToLower()

        '可选语法
        'Dim upperLowerWords = From w In words _
        '                      Select New With {.Upper = w.ToUpper(), .Lower = w.ToLower()}

        For Each ul In upperLowerWords
            Console.WriteLine("Uppercase: " & ul.Upper & ", Lowercase: " & ul.Lower)
        Next
    End Sub

    <Category("投影(Select)")> _
    <Title("Select - 匿名类型 2")> _
    <Description("此示例使用 Select 生成一个序列，其中包含数字的文本" & _
                 "表示形式以及这些数字的长度为偶数还是奇数。")> _
    Public Sub Linq10()
        Dim numbers As Integer() = {5, 4, 1, 3, 9, 8, 6, 7, 2, 0}
        Dim stringNames As String() = {"zero", "one", "two", "three", "four", "five", "six", "seven", "eight", "nine"}

        Dim digitOddEvens = From num In numbers _
                            Select Digit = stringNames(num), Even = ((num Mod 2) = 0)

        For Each d In digitOddEvens
            Console.WriteLine("The digit " & d.Digit & " is " & If(d.Even, "even", "odd"))
        Next
    End Sub

    <Category("投影(Select)")> _
    <Title("Select - 匿名类型 3")> _
    <Description("此示例使用 Select 生成包含" & _
                 "产品某些属性(包括在所得类型中重命名为 Price 的 UnitPrice)" & _
                 "的序列。")> _
    Public Sub Linq11()
        Dim products = GetProductList()

        Dim productInfos = From prod In products _
                           Select prod.ProductName, prod.Category, Price = prod.UnitPrice

        Console.WriteLine("Product Info:")
        For Each prodInfo In productInfos
            Console.WriteLine("{0} is in the category {1} and costs {2} per unit.", _
                              prodInfo.ProductName, prodInfo.Category, prodInfo.Price)
        Next
    End Sub

    <Category("投影(Select)")> _
    <Title("Select - 带索引")> _
    <Description("此示例使用带索引的 Select 子句确定数组中整数的值是否" & _
                 "与其在数组中的位置相符。")> _
    Public Sub Linq12()
        Dim numbers As Integer() = {5, 4, 1, 3, 9, 8, 6, 7, 2, 0}

        Dim numsInPlace = numbers.Select(Function(num, index) New With {.Num = num, .InPlace = (num = index)})

        Console.WriteLine("Number: In-place?")
        For Each n In numsInPlace
            Console.WriteLine(n.Num & ": " & n.InPlace)
        Next
    End Sub

    <Category("投影(Select)")> _
    <Title("Select - 筛选")> _
    <Description("此示例结合使用 select 和 where，以建立一个" & _
                 "返回每个小于 5 的数字的文本形式的简单查询。")> _
    Public Sub Linq13()
        Dim numbers As Integer() = {5, 4, 1, 3, 9, 8, 6, 7, 2, 0}
        Dim digits As String() = {"zero", "one", "two", "three", "four", "five", "six", "seven", "eight", "nine"}

        Dim lowNums = From num In numbers _
                      Where num < 5 _
                      Select digits(num)

        Console.WriteLine("Numbers < 5:")
        For Each num In lowNums
            Console.WriteLine(num)
        Next
    End Sub

    <Category("投影(Select)")> _
    <Title("SelectMany - 复合 From 1")> _
    <Description("此示例使用复合 From 子句进行查询，该查询" & _
                 "返回来自两个数组的所有数字对，其中 numbersA 数字小于" & _
                 "numbersB 数字。")> _
    Public Sub Linq14()
        Dim numbersA() As Integer = {0, 2, 4, 5, 6, 8, 9}
        Dim numbersB() As Integer = {1, 3, 5, 7, 8}

        Dim pairs = From a In numbersA, b In numbersB _
                    Where a < b _
                    Select a, b

        Console.WriteLine("Pairs where a < b:")
        For Each pair In pairs
            Console.WriteLine(pair.a & " is less than " & pair.b)
        Next
    End Sub

    <Category("投影(Select)")> _
    <Title("SelectMany - 复合 From 2")> _
    <Description("此示例使用复合 From 子句选择" & _
                 "订单总价小于 500.00 的所有订单。")> _
    Public Sub Linq15()
        Dim customers = GetCustomerList()

        Dim orders = From cust In customers, ord In cust.Orders _
                     Where ord.Total < 500.0 _
                     Select cust.CustomerID, ord.OrderID, ord.Total

        ObjectDumper.Write(orders)
    End Sub

    <Category("投影(Select)")> _
    <Title("SelectMany - 复合 From 3")> _
    <Description("此示例使用复合 From 子句选择" & _
                 "1998 年或之后下的所有订单。")> _
    Public Sub Linq16()
        Dim customers = GetCustomerList()

        Dim orders = From cust In customers, ord In cust.Orders _
                     Where ord.OrderDate >= #1/1/1998# _
                     Select cust.CustomerID, ord.OrderID, ord.OrderDate

        ObjectDumper.Write(orders)
    End Sub

    <Category("投影(Select)")> _
    <Title("SelectMany - From 赋值")> _
    <Description("此示例使用复合 From 子句选择" & _
                 "订单总价大于 2000.00 的所有订单，并使用 From 赋值以避免" & _
                 "重复请求总价。")> _
    Public Sub Linq17()

        Dim customers = GetCustomerList()

        Dim orders = From cust In customers _
                     From ord In cust.Orders _
                     Where ord.Total >= 2000.0# _
                     Select cust.CustomerID, ord.OrderID, ord.Total

        ObjectDumper.Write(orders)
    End Sub

    <Category("投影(Select)")> _
    <Title("SelectMany - 多个 From")> _
    <Description("此示例使用多个 From 子句，以便可以在选择客户的订单之前" & _
                 "对客户进行筛选。这使得查询更高效，因为" & _
                 "不必选择然后放弃华盛顿以外的客户的订单。")> _
    Public Sub Linq18()
        Dim customers = GetCustomerList()

        Dim cutoffDate = #1/1/1997#

        Dim orders = From cust In customers, ord In cust.Orders _
                     Where cust.Region = "WA" AndAlso ord.OrderDate >= cutoffDate _
                     Select cust.CustomerID, ord.OrderID

        ObjectDumper.Write(orders)
    End Sub

    <Category("投影(Select)")> _
    <Title("SelectMany - 带索引")> _
    <Description("此示例使用带索引的 SelectMany 子句选择所有订单，" & _
                 "同时按从查询返回这些订单的顺序引用" & _
                 "客户。")> _
    Public Sub Linq19()
        Dim customers = GetCustomerList()

        Dim customerOrders = customers.SelectMany(Function(cust, custIndex) _
                                                    From ord In cust.Orders _
                                                    Select "Customer #" & (custIndex + 1) & _
                                                           " has an order with OrderID " & ord.OrderID)
        ObjectDumper.Write(customerOrders)
    End Sub

    <Category("分区(Skip/Take)")> _
    <Title("Take - 简单用法")> _
    <Description("此示例使用 Take 获取数组中的前 3 个" & _
                 "元素。")> _
    Public Sub Linq20()
        Dim numbers() As Integer = {5, 4, 1, 3, 9, 8, 6, 7, 2, 0}

        Dim first3Numbers = From num In numbers Take 3

        Console.WriteLine("First 3 numbers:")
        For Each n In first3Numbers
            Console.WriteLine(n)
        Next
    End Sub

    <Category("分区(Skip/Take)")> _
    <Title("Take - 嵌套")> _
    <Description("此示例使用 Take 获取华盛顿客户的前 3 个" & _
                 "订单。")> _
    Public Sub Linq21()
        Dim customers = GetCustomerList()

        Dim first3WAOrders = From cust In customers, ord In cust.Orders _
                             Where cust.Region = "WA" _
                             Select cust.CustomerID, ord.OrderID, ord.OrderDate _
                             Take 3

        Console.WriteLine("First 3 orders in WA:")

        ObjectDumper.Write(first3WAOrders)
    End Sub

    <Category("分区(Skip/Take)")> _
    <Title("Skip - 简单用法")> _
    <Description("此示例使用 Skip 获取数组中除前 4 个元素以外的" & _
                 "所有元素。")> _
    Public Sub Linq22()
        Dim numbers() As Integer = {5, 4, 1, 3, 9, 8, 6, 7, 2, 0}

        Dim allButFirst4Numbers = From num In numbers Skip 4

        Console.WriteLine("All but first 4 numbers:")
        For Each n In allButFirst4Numbers
            Console.WriteLine(n)
        Next
    End Sub

    <Category("分区(Skip/Take)")> _
    <Title("Skip - 嵌套")> _
    <Description("此示例使用 Skip 获取除华盛顿客户前 2 个订单以外" & _
                 "的所有订单。")> _
    Public Sub Linq23()
        Dim customers = GetCustomerList()

        Dim allButFirst2Orders = From cust In customers, ord In cust.Orders _
                                 Where cust.Region = "WA" _
                                 Select cust.CustomerID, ord.OrderID, ord.OrderDate _
                                 Skip 2


        Console.WriteLine("All but first 2 orders in WA:")

        ObjectDumper.Write(allButFirst2Orders)
    End Sub

    <Category("分区(Skip/Take)")> _
    <Title("Take While - 简单用法")> _
    <Description("此示例使用 Take While 返回从数组开头起的所有元素，" & _
                 "直到碰到不小于 6 的数字。")> _
    Public Sub Linq24()
        Dim numbers() As Integer = {5, 4, 1, 3, 9, 8, 6, 7, 2, 0}

        Dim firstNumbersLessThan6 = From num In numbers Take While num < 6

        '可选语法
        'Dim firstNumbersLessThan6 = numbers.TakeWhile(Function(n) n < 6)

        Console.WriteLine("First numbers less than 6:")
        For Each n In firstNumbersLessThan6
            Console.WriteLine(n)
        Next
    End Sub

    <Category("分区(Skip/Take)")> _
    <Title("Take While - 带索引")> _
    <Description("此示例使用 TakeWhile 返回从数组开头起的所有元素，" & _
                "直到碰到小于自身在数组中位置" & _
                "的数字。")> _
    Public Sub Linq25()
        Dim numbers() As Integer = {5, 4, 1, 3, 9, 8, 6, 7, 2, 0}

        Dim firstSmallNumbers = numbers.TakeWhile(Function(n, index) n >= index)

        Console.WriteLine("First numbers not less than their position:")
        For Each n In firstSmallNumbers
            Console.WriteLine(n)
        Next
    End Sub

    <Category("分区(Skip/Take)")> _
    <Title("Skip While - 简单用法")> _
    <Description("此示例使用 Skip While 获取数组中从第一个" & _
                 "能被 3 整除的元素开始的所有元素。")> _
    Public Sub Linq26()
        Dim numbers() As Integer = {5, 4, 1, 3, 9, 8, 6, 7, 2, 0}
        Dim allButFirst3Numbers = From num In numbers _
                                  Skip While num Mod 3 <> 0

        Console.WriteLine("All elements starting from first element divisible by 3:")
        For Each n In allButFirst3Numbers
            Console.WriteLine(n)
        Next
    End Sub

    <Category("分区(Skip/Take)")> _
    <Title("Skip While - 带索引")> _
    <Description("此示例使用 SkipWhile 获取数组中从第一个" & _
                 "小于自身位置的元素开始的所有元素。")> _
    Public Sub Linq27()
        Dim numbers() As Integer = {5, 4, 1, 3, 9, 8, 6, 7, 2, 0}

        Dim laterNumbers = numbers.SkipWhile(Function(n, index) n >= index)

        Console.WriteLine("All elements starting from first element less than its position:")
        For Each n In laterNumbers
            Console.WriteLine(n)
        Next
    End Sub


    <Category("排序(Order By)")> _
    <Title("Order By - 简单用法 1")> _
    <Description("此示例使用 Order By 按字母顺序对单词列表进行排序。")> _
    Public Sub Linq28()
        Dim words() As String = {"cherry", "apple", "blueberry"}

        Dim sortedWords = From word In words _
                          Order By word

        Console.WriteLine("The sorted list of words:")
        For Each w In sortedWords
            Console.WriteLine(w)
        Next
    End Sub

    <Category("排序(Order By)")> _
    <Title("Order By - 简单用法 2")> _
    <Description("此示例使用 Order By 按长度对单词列表进行排序。")> _
    Public Sub Linq29()
        Dim words = New String() {"cherry", "apple", "blueberry"}

        Dim sortedWords = From word In words _
                          Order By word.Length

        Console.WriteLine("The sorted list of words (by length):")
        For Each w In sortedWords
            Console.WriteLine(w)
        Next

    End Sub

    <Category("排序(Order By)")> _
    <Title("Order By - 简单用法 3")> _
    <Description("此示例使用 Order By 按名称对产品列表进行排序。")> _
    Public Sub Linq30()
        Dim products = GetProductList()

        Dim sortedProducts = From prod In products _
                             Order By prod.ProductName

        ObjectDumper.Write(sortedProducts)
    End Sub

    Public Class CaseInsensitiveComparer : Implements IComparer(Of String)

        Public Function Compare(ByVal x As String, ByVal y As String) As Integer Implements IComparer(Of String).Compare
            Return String.Compare(x, y, True)
        End Function
    End Class

    <Category("排序(Order By)")> _
    <Title("Order By - 比较器")> _
    <Description("此示例使用 Order By 和自定义比较器" & _
                 "对数组中的单词进行不区分大小写的排序。")> _
    <LinkedClass("CaseInsensitiveComparer")> _
    Public Sub Linq31()
        Dim words = New String() {"aPPLE", "AbAcUs", "bRaNcH", "BlUeBeRrY", "ClOvEr", "cHeRry"}

        Dim sortedWords = words.OrderBy(Function(word) word, New CaseInsensitiveComparer())

        ObjectDumper.Write(sortedWords)
    End Sub


    <Category("排序(Order By)")> _
    <Title("Order By Descending - 简单用法 1")> _
    <Description("此示例使用 Order By 和 Descending 从高到低" & _
                 "对 double 类型数据的列表进行排序。")> _
    Public Sub Linq32()
        Dim doubles = New Double() {1.7, 2.3, 1.9, 4.1, 2.9}

        Dim sortedDoubles = From dbl In doubles _
                            Order By dbl Descending

        Console.WriteLine("The doubles from highest to lowest:")
        For Each d In sortedDoubles
            Console.WriteLine(d)
        Next
    End Sub

    <Category("排序(Order By)")> _
    <Title("Order By Descending - 简单用法 2")> _
    <Description("此示例使用 Order By 按库存量" & _
                 "从高到低对产品列表进行排序。")> _
    Public Sub Linq33()
        Dim products = GetProductList()

        Dim sortedProducts = From prod In products _
                             Order By prod.UnitsInStock Descending

        ObjectDumper.Write(sortedProducts)
    End Sub

    <Category("排序(Order By)")> _
    <Title("Order By Descending - 比较器")> _
    <Description("此示例使用 Order By 和自定义比较器" & _
                 "对数组中的单词进行不区分大小写的降序排序。")> _
    <LinkedClass("CaseInsensitiveComparer")> _
    Public Sub Linq34()
        Dim words = New String() {"aPPLE", "AbAcUs", "bRaNcH", "BlUeBeRrY", "ClOvEr", "cHeRry"}

        Dim sortedWords = words.OrderByDescending(Function(word) word, New CaseInsensitiveComparer())

        ObjectDumper.Write(sortedWords)
    End Sub


    <Category("排序(Order By)")> _
    <Title("复合 Order By")> _
    <Description("此示例使用复合 Order By 对数字列表进行排序，" & _
                 "先按其名称长度排序，再按名称自身的字母顺序排序。")> _
    Public Sub Linq35()
        Dim digits = New String() {"zero", "one", "two", "three", "four", "five", "six", "seven", "eight", "nine"}

        Dim sortedDigits = From dig In digits _
                           Order By dig.Length, dig

        Console.WriteLine("Sorted digits:")
        For Each d In sortedDigits
            Console.WriteLine(d)
        Next
    End Sub

    <Category("排序(Order By)")> _
    <Title("复合 Order By - 先升序再降序")> _
    <Description("此示例使用复合 Order By 先按单词长度排序，" & _
                 "再按名称自身的字母顺序降序排序。")> _
    <LinkedClass("CaseInsensitiveComparer")> _
    Public Sub Linq36()
        Dim words() As String = {"aPPLE", "AbAcUs", "bRaNcH", "BlUeBeRrY", "ClOvEr", "cHeRry"}

        Dim sortedWords = From word In words _
                          Order By word.Length, word Descending

        ObjectDumper.Write(sortedWords)
    End Sub

    <Category("排序(Order By)")> _
    <Title("复合 Order By - 先降序再升序")> _
    <Description("此示例使用复合 Order By 对产品列表" & _
                 "先按类别(降序)排序，再按单价排序。")> _
    Public Sub Linq37()
        Dim products = GetProductList()

        Dim sortedProducts = From prod In products _
                             Order By prod.Category Descending, prod.UnitPrice

        ObjectDumper.Write(sortedProducts)
    End Sub

    <Category("排序(Order By)")> _
    <Title("ThenByDescending - 比较器")> _
    <Description("此示例使用 OrderBy 子句和 ThenBy 子句及自定义比较器" & _
                 "对数组中的单词先按单词长度排序，" & _
                 "再进行不区分大小写的降序排序。")> _
    <LinkedClass("CaseInsensitiveComparer")> _
    Public Sub Linq38()
        Dim words() As String = {"aPPLE", "AbAcUs", "bRaNcH", "BlUeBeRrY", "ClOvEr", "cHeRry"}

        Dim sortedWords = words.OrderBy(Function(word) word.Length) _
                               .ThenByDescending(Function(word) word, New CaseInsensitiveComparer())

        ObjectDumper.Write(sortedWords)
    End Sub

    <Category("排序(Order By)")> _
    <Title("Reverse")> _
    <Description("此示例使用 Reverse 将数组中所有第二个字母为“i”的数字，" & _
                 "以其在原始数组中的相反顺序创建列表。")> _
    Public Sub Linq39()
        Dim digits() As String = {"zero", "one", "two", "three", "four", "five", "six", "seven", "eight", "nine"}

        Dim IDigits = From dig In digits _
                      Where dig(1) = "i"

        Dim reversedIDigits = IDigits.Reverse()

        Console.WriteLine("A backwards list of the digits with a second character of 'i':")
        For Each d In reversedIDigits
            Console.WriteLine(d)
        Next
    End Sub

    Public Class AnagramEqualityComparer
        Implements IEqualityComparer(Of String)

        Public Shadows Function Equals(ByVal x As String, ByVal y As String) As Boolean Implements IEqualityComparer(Of String).Equals
            Return getCanonicalString(x) = getCanonicalString(y)
        End Function

        Public Shadows Function GetHashCode(ByVal obj As String) As Integer Implements IEqualityComparer(Of String).GetHashCode
            Return getCanonicalString(obj).GetHashCode()
        End Function

        Private Function getCanonicalString(ByVal word As String) As String
            Dim wordChars = word.ToCharArray()
            Array.Sort(wordChars)
            Return New String(wordChars)
        End Function

    End Class

    <Category("分组(Group By)")> _
    <Title("Group By - 简单用法 1")> _
    <Description("此示例使用 Group By 按" & _
                 "数字除以 5 所得的余数划分数字列表。")> _
    Public Sub Linq40()
        Dim numbers() As Integer = {5, 4, 1, 3, 9, 8, 6, 7, 2, 0}

        Dim numberGroups = From num In numbers _
                           Group num By Remainder = num Mod 5 Into NumberGroup = Group _
                           Select Remainder, NumberGroup

        For Each g In numberGroups
            Console.WriteLine("Numbers with a remainder of " & g.Remainder & "when divided by 5:")
            For Each n In g.NumberGroup
                Console.WriteLine(n)
            Next
        Next
    End Sub

    <Category("分组(Group By)")> _
    <Title("Group By - 简单用法 2")> _
    <Description("此示例使用 Group By 按" & _
                 "单词的首字母划分单词列表。")> _
    Public Sub Linq41()
        Dim words = New String() {"blueberry", "chimpanzee", "abacus", "banana", "apple", "cheese"}

        Dim wordGroups = From word In words _
                         Group word By Key = word(0) Into Group _
                         Select FirstLetter = Key, WordGroup = Group

        For Each g In wordGroups
            Console.WriteLine("Words that start with the letter '" & g.FirstLetter & "':")
            For Each w In g.WordGroup
                Console.WriteLine(w)
            Next
        Next

    End Sub

    <Category("分组(Group By)")> _
    <Title("Group By - 简单用法 3")> _
    <Description("此示例使用 Group By 按类别划分产品列表。")> _
    Public Sub Linq42()
        Dim products = GetProductList()

        Dim orderGroups = From prod In products _
                          Group prod By prod.Category Into Group _
                          Select Category, ProductGroup = Group

        ObjectDumper.Write(orderGroups, 1)
    End Sub

    <Category("分组(Group By)")> _
    <Title("Group By - 嵌套")> _
    <Description("此示例使用 Group By 先按年再按月" & _
                 "划分每个客户的订单列表。")> _
    Public Sub Linq43()
        Dim customers = GetCustomerList()

        Dim custOrderGroups = From cust In customers _
                              Select cust.CompanyName, _
                                     Groups = (From ord In cust.Orders _
                                               Group By ord.OrderDate.Year, ord.OrderDate.Month _
                                               Into Group)

        ObjectDumper.Write(custOrderGroups, 3)
    End Sub

    <Category("分组(Group By)")> _
    <Title("Group By - 比较器")> _
    <Description("此示例使用 GroupBy 以及使互为变位词的单词相匹配的自定义比较器来" & _
                 "划分数组中经过裁剪的元素。")> _
    <LinkedClass("AnagramEqualityComparer")> _
    Public Sub Linq44()
        Dim anagrams() As String = {"from   ", " salt", " earn ", "  last   ", " near ", " form  "}

        Dim orderGroups = anagrams.GroupBy(Function(word) word.Trim(), New AnagramEqualityComparer())

        ObjectDumper.Write(orderGroups, 1)
    End Sub

    <Category("分组(Group By)")> _
    <Title("Group By - 比较器、映射")> _
    <Description("此示例使用 GroupBy 以及使互为变位词的单词相匹配的自定义比较器来" & _
                 "划分数组中经过裁剪的元素。" & _
                 "然后将结果转换为大写。")> _
    <LinkedClass("AnagramEqualityComparer")> _
    Public Sub Linq45()
        Dim anagrams() As String = {"from   ", " salt", " earn ", "  last   ", " near ", " form  "}

        Dim orderGroups = anagrams.GroupBy(Function(word) word.Trim(), _
                                           Function(word) word.ToUpper(), _
                                           New AnagramEqualityComparer())

        ObjectDumper.Write(orderGroups, 1)
    End Sub

    <Category("集合运算符(Distinct/Union...)")> _
    <Title("Distinct - 1")> _
    <Description("此示例使用 Distinct 移除 300 的因子" & _
                "序列中重复的元素。")> _
    Public Sub Linq46()
        Dim factorsOf300 = New Integer() {2, 2, 3, 5, 5}

        Dim uniqueFactors = From factor In factorsOf300 Distinct

        Console.WriteLine("Prime factors of 300:")
        For Each f In uniqueFactors
            Console.WriteLine(f)
        Next
    End Sub

    <Category("集合运算符(Distinct/Union...)")> _
    <Title("Distinct - 2")> _
    <Description("此示例使用 Distinct 查找唯一类别名称。")> _
    Public Sub Linq47()
        Dim products = GetProductList()

        Dim categoryNames = From prod In products _
                            Select prod.Category _
                            Distinct

        Console.WriteLine("Category names:")
        For Each n In categoryNames
            Console.WriteLine(n)
        Next
    End Sub

    <Category("集合运算符(Distinct/Union...)")> _
    <Title("Distinct 带 Key")> _
    <Description("此示例使用 Distinct 只选择每个国家/地区一次，并" & _
                 "选取一个公司。Distinct 运算符将仅对" & _
                 "Country 字段起作用，因为该字段已被标以“Key”修饰符。")> _
    Public Sub Linq47b()
        Dim customers = GetCustomerList()

        '将 Country 标为 Key 字段表示匿名类型将替代 
        'Equals 方法，以便仅比较 Country 字段(而非比较所有字段)
        '因此，当 Distinct 调用 Equals 时将仅考虑 Country
        '字段。
        Dim cities = From cust In customers _
                     Select New With {Key cust.Country, cust.CompanyName} _
                     Distinct

        '请注意，通过自行编写自定义比较器
        '并将其传递给 Distinct 扩展方法，可以得到相同结果，但使用 Key 修饰符
        '会简便许多

        ObjectDumper.Write(cities)
    End Sub


    <Category("集合运算符(Distinct/Union...)")> _
    <Title("Union - 1")> _
    <Description("此示例使用 Union 创建一个序列，其中包含" & _
                 "来自两个数组的唯一值。")> _
    Public Sub Linq48()
        Dim numbersA() As Integer = {0, 2, 4, 5, 6, 8, 9}
        Dim numbersB() As Integer = {1, 3, 5, 7, 8}

        Dim uniqueNumbers = numbersA.Union(numbersB)

        Console.WriteLine("Unique numbers from both arrays:")
        For Each n In uniqueNumbers
            Console.WriteLine(n)
        Next
    End Sub

    <Category("集合运算符(Distinct/Union...)")> _
    <Title("Union - 2")> _
    <Description("此示例使用 Union 创建一个序列，其中包含" & _
                 "来自产品名和客户名的唯一首字母。")> _
    Public Sub Linq49()
        Dim products = GetProductList()
        Dim customers = GetCustomerList()

        Dim productFirstChars = From prod In products _
                                Select prod.ProductName(0)

        Dim customerFirstChars = From cust In customers _
                                 Select cust.CompanyName(0)

        Dim uniqueFirstChars = productFirstChars.Union(customerFirstChars)

        Console.WriteLine("Unique first letters from Product names and Customer names:")
        For Each ch In uniqueFirstChars
            Console.WriteLine(ch)
        Next
    End Sub

    <Category("集合运算符(Distinct/Union...)")> _
    <Title("Intersect - 1")> _
    <Description("此示例使用 Intersect 创建一个序列，其中包含" & _
                 "两个数组所共有的值。")> _
    Public Sub Linq50()
        Dim numbersA() As Integer = {0, 2, 4, 5, 6, 8, 9}
        Dim numbersB() As Integer = {1, 3, 5, 7, 8}

        Dim commonNumbers = numbersA.Intersect(numbersB)

        Console.WriteLine("Common numbers shared by both arrays:")
        For Each n In commonNumbers
            Console.WriteLine(n)
        Next
    End Sub

    <Category("集合运算符(Distinct/Union...)")> _
    <Title("Intersect - 2")> _
    <Description("此示例使用 Intersect 创建一个序列，其中包含" & _
                 "产品名和客户名所共有的首字母。")> _
    Public Sub Linq51()
        Dim products = GetProductList()
        Dim customers = GetCustomerList()

        Dim productFirstChars = From prod In products _
                                Select prod.ProductName(0)

        Dim customerFirstChars = From cust In customers _
                                 Select cust.CompanyName(0)

        Dim commonFirstChars = productFirstChars.Intersect(customerFirstChars)

        Console.WriteLine("Common first letters from Product names and Customer names:")
        For Each ch In commonFirstChars
            Console.WriteLine(ch)
        Next
    End Sub

    <Category("集合运算符(Distinct/Union...)")> _
    <Title("Except - 1")> _
    <Description("此示例使用 Except 创建一个序列，其中包含 numbersA 中有" & _
                 "但 numbersB 中没有的值。")> _
    Public Sub Linq52()
        Dim numbersA() As Integer = {0, 2, 4, 5, 6, 8, 9}
        Dim numbersB() As Integer = {1, 3, 5, 7, 8}

        Dim aOnlyNumbers = numbersA.Except(numbersB)

        Console.WriteLine("Numbers in first array but not second array:")
        For Each n In aOnlyNumbers
            Console.WriteLine(n)
        Next
    End Sub

    <Category("集合运算符(Distinct/Union...)")> _
    <Title("Except - 2")> _
    <Description("此示例使用 Except 创建一个序列，其中包含" & _
                 "并非也是客户名首字母的产品名首字母。")> _
    Public Sub Linq53()
        Dim products = GetProductList()
        Dim customers = GetCustomerList()

        Dim productFirstChars = From prod In products _
                                Select prod.ProductName(0)

        Dim customerFirstChars = From cust In customers _
                                 Select cust.CompanyName(0)

        Dim productOnlyFirstChars = productFirstChars.Except(customerFirstChars)

        Console.WriteLine("First letters from Product names, but not from Customer names:")
        For Each ch In productOnlyFirstChars
            Console.WriteLine(ch)
        Next
    End Sub

    <Category("转换(ToArray/ToList...)")> _
    <Title("ToArray")> _
    <Description("此示例使用 ToArray 将序列直接计算为数组。")> _
    Public Sub Linq54()
        Dim doubles() As Double = {1.7, 2.3, 1.9, 4.1, 2.9}

        Dim sortedDoubles = From dbl In doubles _
                            Order By dbl Descending

        Dim doublesArray = sortedDoubles.ToArray()

        Console.WriteLine("Every other double from highest to lowest:")
        For d As Integer = 0 To doublesArray.Length
            Console.WriteLine(doublesArray(d))
            d += 1
        Next
    End Sub

    <Category("转换(ToArray/ToList...)")> _
    <Title("ToList")> _
    <Description("此示例使用 ToList 将序列直接计算为 List(Of T)。")> _
    Public Sub Linq55()
        Dim words() As String = {"cherry", "apple", "blueberry"}

        Dim sortedWords = From word In words _
                          Order By word

        Dim wordList = sortedWords.ToList()

        Console.WriteLine("The sorted word list:")
        For Each w In wordList
            Console.WriteLine(w)
        Next
    End Sub

    <Category("转换(ToArray/ToList...)")> _
    <Title("ToDictionary")> _
    <Description("此示例使用 ToDictionary 将序列和" & _
                 "相关的键表达式直接计算为字典。")> _
    Public Sub Linq56()
        Dim scoreRecords = MakeList(New With {.Name = "Alice", .Score = 50}, _
                                    New With {.Name = "Bob", .Score = 40}, _
                                    New With {.Name = "Cathy", .Score = 45})

        Dim scoreRecordsDict = scoreRecords.ToDictionary(Function(sr) sr.Name)

        Console.WriteLine("Bob's score: " & scoreRecordsDict("Bob").ToString)

    End Sub

    <Category("转换(ToArray/ToList...)")> _
    <Title("TypeOf")> _
    <Description("此示例使用 TypeOf 仅返回数组中 double 类型的元素。")> _
    Public Sub Linq57()
        Dim numbers = New Object() {Nothing, 1.0, "two", 3, "four", 5, "six", 7.0}

        Dim doubles = From num In numbers _
                      Where TypeOf num Is Double

        Console.WriteLine("Numbers stored as doubles:")
        For Each d In doubles
            Console.WriteLine(d)
        Next
    End Sub

    <Category("元素运算符(First/ElementAt...)")> _
    <Title("First - 简单用法")> _
    <Description("此示例使用 First 将第一个匹配元素" & _
                 "作为产品返回，而非作为包含产品的序列返回。")> _
    Public Sub Linq58()
        Dim products = GetProductList()

        Dim product12 = From prod In products _
                        Where prod.ProductID = 12 _
                        Take 1

        ObjectDumper.Write(product12)
    End Sub

    <Category("元素运算符(First/ElementAt...)")> _
    <Title("First - 条件")> _
    <Description("此示例使用 First 查找数组中第一个以“o”开头的元素。")> _
    Public Sub Linq59()

        Dim strings = New String() {"zero", "one", "two", "three", "four", "five", "six", "seven", "eight", "nine"}

        Dim FirstO = Aggregate str In strings _
                     Into First(str(0) = "o"c)

        '可选语法
        'Dim FirstO = strings.First(Function(s) s(0) = "o"c)

        Console.WriteLine("A string starting with 'o': " & FirstO)
    End Sub

    <Category("元素运算符(First/ElementAt...)")> _
    <Title("FirstOrDefault - 简单用法")> _
    <Description("此示例使用 FirstOrDefault 尝试返回序列的第一个元素，" & _
                 "如果没有元素，则返回该类型的" & _
                 "默认值。")> _
    Public Sub Linq61()
        Dim numbers As Integer() = {}

        '数组中没有元素，因此返回默认值(zero)
        Dim firstNumOrDefault = numbers.FirstOrDefault()

        Console.WriteLine(firstNumOrDefault)
    End Sub

    <Category("元素运算符(First/ElementAt...)")> _
    <Title("FirstOrDefault - 条件")> _
    <Description("此示例使用 FirstOrDefault 将 ProductID 为 789  的第一种产品作为" & _
                 "单个 Product 对象返回，如果没有匹配项，则返回 Nothing。")> _
    Public Sub Linq62()
        Dim products = GetProductList()

        '这样将不会返回任何结果，因为不存在 ID 为 789 的产品
        Dim product789 = Aggregate prod In products _
                         Into FirstOrDefault(prod.ProductID = 789)

        '可选语法
        'Dim product789 = products.FirstOrDefault(Function(p) p.ProductID = 789)
        Console.WriteLine("Product 789 exists: " & (product789 IsNot Nothing))
    End Sub


    <Category("元素运算符(First/ElementAt...)")> _
    <Title("ElementAt")> _
    <Description("此示例使用 ElementAt 从数组检索第二个大于 5 " & _
                 "的数字。")> _
    Public Sub Linq64()
        Dim numbers() As Integer = {5, 4, 1, 3, 9, 8, 6, 7, 2, 0}

        '第二个数字的索引是 1，因为序列的索引从 0 开始
        Dim fourthLowNum = (From num In numbers Where num > 5).ElementAt(1)

        Console.WriteLine("Second number > 5: " & fourthLowNum)
    End Sub

    <Category("生成运算符(Range/Repeat)")> _
    <Title("Range")> _
    <Description("此示例使用 Range 生成一个从 100 到 149 的数字序列，" & _
                 "用于查找该范围内的哪些数字是奇数和偶数。")> _
    Public Sub Linq65()

        '使用新的三元 If 运算符返回“odd”或“even”
        Dim numbers = From num In Enumerable.Range(100, 50) _
                      Select Number = num, OddEven = If(num Mod 2 = 1, "odd", "even")

        For Each n In numbers
            Console.WriteLine("The number " & n.Number & " is " & n.OddEven)
        Next

    End Sub

    <Category("生成运算符(Range/Repeat)")> _
    <Title("Repeat")> _
    <Description("此示例使用 Repeat 生成包含数字 7 十次的序列。")> _
    Public Sub Linq66()

        Dim numbers = Enumerable.Repeat(7, 10)

        For Each n In numbers
            Console.WriteLine(n)
        Next
    End Sub


    <Category("限定符(Any/All)")> _
    <Title("Any - 简单用法")> _
    <Description("此示例使用 Any 确定数组中是否有任何单词" & _
                 "包含子字符串“ei”。")> _
    Public Sub Linq67()

        Dim words() As String = {"believe", "relief", "receipt", "field"}

        Dim iAfterE = Aggregate word In words _
                      Into Any(word.Contains("ei"))

        '可选语法
        'Dim iAfterE = words.Any(Function(w) w.Contains("ei"))

        Console.WriteLine("There is a word that contains in the list that contains 'ei': " & iAfterE)
    End Sub


    <Category("限定符(Any/All)")> _
    <Title("Any - 分组")> _
    <Description("此示例使用 Any 仅对至少有一种产品缺货的类别" & _
                 "返回经过分组的产品列表。")> _
    Public Sub Linq69()
        Dim products = GetProductList()

        Dim productGroups = From prod In products _
                            Group prod By prod.Category Into Group _
                            Where Group.Any(Function(p) p.UnitsInStock = 0) _
                            Select Category, ProductGroup = Group

        ObjectDumper.Write(productGroups, 1)
    End Sub

    <Category("限定符(Any/All)")> _
    <Title("All - 简单用法")> _
    <Description("此示例使用 All 确定数组是否仅包含" & _
                 "奇数。")> _
    Public Sub Linq70()
        Dim numbers() As Integer = {1, 11, 3, 19, 41, 65, 19}

        Dim onlyOdd = Aggregate num In numbers _
                      Into All(num Mod 2 = 1)

        Console.WriteLine("The list contains only odd numbers: " & onlyOdd)
    End Sub

    <Category("限定符(Any/All)")> _
    <Title("All - 分组")> _
    <Description("此示例使用 All 仅对所有产品都有货的类别" & _
                 "返回经过分组的产品列表。")> _
    Public Sub Linq72()

        Dim products = GetProductList()

        Dim productGroups = From prod In products _
                            Group By prod.Category _
                            Into ProductGroup = Group, _
                                 AllUnitsInStock = All(prod.UnitsInStock > 0) _
                            Where AllUnitsInStock = True _
                            Select Category, ProductGroup

        '可选语法
        'Dim productGroups = From p In products _
        '                    Group p By p.Category Into Group _
        '                    Where (Group.All(Function(p) p.UnitsInStock > 0)) _
        '                    Select Category, ProductGroup = Group

        ObjectDumper.Write(productGroups, 1)
    End Sub

    <Category("聚合(Min/Max/Count...)")> _
    <Title("Count - 简单用法")> _
    <Description("此示例使用 Count 获取 300 的唯一因子的数量。")> _
    Public Sub Linq73()
        Dim factorsOf300 = New Integer() {2, 2, 3, 5, 5}

        Dim uniqueFactors = factorsOf300.Distinct().Count()

        Console.WriteLine("There are " & uniqueFactors & " unique factors of 300.")
    End Sub

    <Category("聚合(Min/Max/Count...)")> _
    <Title("Count - 条件")> _
    <Description("此示例使用 Count 获取数组中奇数整数的数量。")> _
    Public Sub Linq74()
        Dim numbers() As Integer = {5, 4, 1, 3, 9, 8, 6, 7, 2, 0}

        Dim oddNumbers = Aggregate num In numbers _
                         Into Count(num Mod 2 = 1)

        Console.WriteLine("There are " & oddNumbers & " odd numbers in the list.")
    End Sub



    <Category("聚合(Min/Max/Count...)")> _
    <Title("Count - 嵌套")> _
    <Description("此示例使用 Count 返回客户和每个客户所下订单数" & _
                 "的列表。")> _
    Public Sub Linq76()
        Dim customers = GetCustomerList()

        Dim orderCounts = From cust In customers _
                          Select cust.CustomerID, OrderCount = cust.Orders.Count()

        ObjectDumper.Write(orderCounts)
    End Sub

    <Category("聚合(Min/Max/Count...)")> _
    <Title("Count - 分组")> _
    <Description("此示例使用 Count 返回类别和每个类别所具有产品数" & _
                 "的列表。")> _
    Public Sub Linq77()
        Dim products = GetProductList()

        Dim categoryCounts = From prod In products _
                             Group prod By prod.Category Into Count() _
                             Select Category, ProductCount = Count

        ObjectDumper.Write(categoryCounts)
    End Sub

    <Category("聚合(Min/Max/Count...)")> _
    <Title("Sum - 简单用法")> _
    <Description("此示例使用 Sum 获取数组中数字的总和。")> _
    Public Sub Linq78()

        Dim numbers() As Integer = {5, 4, 1, 3, 9, 8, 6, 7, 2, 0}

        Dim numSum = numbers.Sum()

        Console.WriteLine("The sum of the numbers is " & numSum)
    End Sub

    <Category("聚合(Min/Max/Count...)")> _
    <Title("Sum - 投影")> _
    <Description("此示例使用 Sum 获取数组中所有单词的" & _
                 "总字符数。")> _
    Public Sub Linq79()
        Dim words = New String() {"cherry", "apple", "blueberry"}

        Dim totalChars = Aggregate word In words _
                         Into Sum(word.Length)

        Console.WriteLine("There are a total of " & totalChars & " characters in these words.")
    End Sub

    <Category("聚合(Min/Max/Count...)")> _
    <Title("Sum - 分组")> _
    <Description("此示例使用 Sum 获取每个产品类别的总库存量。")> _
    Public Sub Linq80()
        Dim products = GetProductList()

        Dim categories = From prod In products _
                         Group prod By prod.Category _
                         Into TotalUnitsInStock = Sum(prod.UnitsInStock)

        '可选语法
        'Dim categories = From p In products _
        '                 Group p By p.Category Into pGroup = Group _
        '                 Select Category, _
        '                        TotalUnitsInStock = pGroup.Sum(Function(p) p.UnitsInStock)

        ObjectDumper.Write(categories)
    End Sub


    <Category("聚合(Min/Max/Count...)")> _
    <Title("Min - 简单用法")> _
    <Description("此示例使用 Min 获取数组中最小的数字。")> _
    Public Sub Linq81()
        Dim numbers() As Integer = {5, 4, 1, 3, 9, 8, 6, 7, 2, 0}

        Dim minNum = numbers.Min()

        Console.WriteLine("The minimum number is " & minNum & ".")
    End Sub

    <Category("聚合(Min/Max/Count...)")> _
    <Title("Min - 投影")> _
    <Description("此示例使用 Min 获取数组中最短单词的长度。")> _
    Public Sub Linq82()
        Dim words = New String() {"cherry", "apple", "blueberry"}

        Dim shortestWord = Aggregate word In words _
                           Into Min(word.Length)

        Console.WriteLine("The shortest word is " & shortestWord & " characters long.")
    End Sub

    <Category("聚合(Min/Max/Count...)")> _
    <Title("Min - 分组")> _
    <Description("此示例使用 Min 获取每个类别中产品的最低价。")> _
    Public Sub Linq83()
        Dim products = GetProductList()

        Dim categories = From prod In products _
                         Group prod By prod.Category _
                         Into CheapestPrice = Min(prod.UnitPrice) _
                         Select Category, CheapestPrice

        ObjectDumper.Write(categories)
    End Sub

    <Category("聚合(Min/Max/Count...)")> _
    <Title("Min - 元素")> _
    <Description("此示例使用 Min 获取每个类别中最低价的产品。")> _
    Public Sub Linq84()
        Dim products = GetProductList()

        Dim categories = From prod In products _
                         Group prod By prod.Category _
                         Into Prods = Group, minPrice = Min(prod.UnitPrice) _
                         Let CheapestProducts = (From p In Prods _
                                                 Where p.UnitPrice = minPrice) _
                         Select Category, CheapestProducts

        ObjectDumper.Write(categories, 1)
    End Sub

    <Category("聚合(Min/Max/Count...)")> _
    <Title("Max - 简单用法")> _
    <Description("此示例使用 Max 获取数组中最大的数字。")> _
    Public Sub Linq85()
        Dim numbers() As Integer = {5, 4, 1, 3, 9, 8, 6, 7, 2, 0}

        Dim maxNum = numbers.Max()

        Console.WriteLine("The maximum number is " & maxNum & ".")
    End Sub

    <Category("聚合(Min/Max/Count...)")> _
    <Title("Max - 投影")> _
    <Description("此示例使用 Max 获取数组中最长单词的长度。")> _
    Public Sub Linq86()
        Dim words = New String() {"cherry", "apple", "blueberry"}

        Dim longestLength = Aggregate word In words _
                            Into Max(word.Length)

        Console.WriteLine("The longest word is " & longestLength & " characters long.")
    End Sub

    <Category("聚合(Min/Max/Count...)")> _
    <Title("Max - 分组")> _
    <Description("此示例使用 Max 获取每个类别中产品的最高价。")> _
    Public Sub Linq87()
        Dim products = GetProductList()

        Dim categories = From prod In products _
                         Group prod By prod.Category _
                         Into Max(prod.UnitPrice)

        '可选语法
        'Dim categories = From prod In products _
        '                 Group prod By prod.Category Into Group _
        '                 Select Category, _
        '                        MostExpensivePrice = Aggregate prod In Group _
        '                                             Into Max(prod.UnitPrice)

        ObjectDumper.Write(categories)
    End Sub

    <Category("聚合(Min/Max/Count...)")> _
    <Title("Max - 元素")> _
    <Description("此示例使用 Max 获取每个类别中最高价的产品。")> _
    Public Sub Linq88()
        Dim products = GetProductList()

        Dim categories = From prod In products _
                         Group prod By prod.Category _
                         Into MaxPrice = Max(prod.UnitPrice), Group _
                         Select Category, MostExpensive = From prod In Group _
                                                          Where prod.UnitPrice = MaxPrice

        '可选语法
        'Dim categories = From prod In products _
        '                 Group prod By prod.Category Into Group _
        '                 Let maxPrice = Group.Max(Function(p) p.UnitPrice) _
        '                 Select Category, MostExpensiveProducts = Group.Where(Function(p) p.UnitPrice = maxPrice)

        ObjectDumper.Write(categories, 1)
    End Sub

    <Category("聚合(Min/Max/Count...)")> _
    <Title("Average - 简单用法")> _
    <Description("此示例使用 Average 获取数组中所有数字的平均值。")> _
    Public Sub Linq89()
        Dim numbers() As Integer = {5, 4, 1, 3, 9, 8, 6, 7, 2, 0}

        Dim averageNum = numbers.Average()

        Console.WriteLine("The average number is " & averageNum & ".")
    End Sub

    <Category("聚合(Min/Max/Count...)")> _
    <Title("Average - 投影")> _
    <Description("此示例使用 Average 获取数组中单词的平均长度。")> _
    Public Sub Linq90()
        Dim words() As String = {"cherry", "apple", "blueberry"}

        Dim averageLength = words.Average(Function(w) w.Length)

        Console.WriteLine("The average word length is " & averageLength & " characters.")
    End Sub

    <Category("聚合(Min/Max/Count...)")> _
    <Title("Average - 分组")> _
    <Description("此示例使用 Average 获取每个类别中产品的平均价格。")> _
    Public Sub Linq91()
        Dim products = GetProductList()

        Dim categories = From prod In products _
                         Group prod By prod.Category _
                         Into Average(prod.UnitPrice)

        ObjectDumper.Write(categories)
    End Sub

    <Category("聚合(Min/Max/Count...)")> _
    <Title("Aggregate - 简单用法")> _
    <Description("此示例使用 Aggregate 创建数组的连乘，" & _
                 "计算所有元素的总乘积。")> _
    Public Sub Linq92()
        Dim doubles = New Double() {1.7, 2.3, 1.9, 4.1, 2.9}

        Dim product = doubles.Aggregate(Function(runningProduct, nextFactor) runningProduct * nextFactor)

        Console.WriteLine("Total product of all numbers: " & product)
    End Sub

    <Category("聚合(Min/Max/Count...)")> _
    <Title("Aggregate - 种子")> _
    <Description("此示例使用 Aggregate 创建一个流水账余额， " & _
                 "从最初余额 100 减去每次取出的金额，直到" & _
                 "余额减少到 0 以下为止。")> _
    Public Sub Linq93()
        Dim startBalance = 100.0

        Dim attemptedWithdrawals() As Integer = {20, 10, 40, 50, 10, 70, 30}

        Dim endBalance = _
            attemptedWithdrawals.Aggregate(startBalance, _
                                           Function(balance, nextWithdrawal) _
                                            (If(nextWithdrawal <= balance, (balance - nextWithdrawal), balance)))

        Console.WriteLine("Ending balance: " & endBalance)
    End Sub



    <Category("杂项(Concat/SequenceEqual...)")> _
    <Title("Concat - 1")> _
    <Description("此示例使用 Concat 创建一个序列，其中依次包含" & _
                 "每个数组的值。")> _
    Public Sub Linq94()
        Dim numbersA() As Integer = {0, 2, 4, 5, 6, 8, 9}
        Dim numbersB() As Integer = {1, 3, 5, 7, 8}

        Dim allNumbers = numbersA.Concat(numbersB)

        Console.WriteLine("All numbers from both arrays:")
        For Each n In allNumbers
            Console.WriteLine(n)
        Next
    End Sub

    <Category("杂项(Concat/SequenceEqual...)")> _
    <Title("Concat - 2")> _
    <Description("此示例使用 Concat 创建一个含有" & _
                 "所有客户名和产品名的序列，包括任何重复项。")> _
    Public Sub Linq95()
        Dim customers = GetCustomerList()
        Dim products = GetProductList()

        Dim customerNames = From cust In customers _
                            Select cust.CompanyName

        Dim productNames = From prod In products _
                           Select prod.ProductName

        Dim allNames = customerNames.Concat(productNames)

        Console.WriteLine("Customer and product names:")
        For Each n In allNames
            Console.WriteLine(n)
        Next
    End Sub

    <Category("杂项(Concat/SequenceEqual...)")> _
    <Title("EqualAll - 1")> _
    <Description("此示例使用 EqualAll 查看两个序列中是否所有元素" & _
                 "以相同顺序匹配。")> _
    Public Sub Linq96()
        Dim wordsA() As String = {"cherry", "apple", "blueberry"}
        Dim wordsB() As String = {"cherry", "apple", "blueberry"}

        Dim match = wordsA.SequenceEqual(wordsB)

        Console.WriteLine("The sequences match: " & match)
    End Sub

    <Category("杂项(Concat/SequenceEqual...)")> _
    <Title("EqualAll - 2")> _
    <Description("此示例使用 EqualAll 查看两个序列中是否所有元素" & _
                 "以相同顺序匹配。")> _
    Public Sub Linq97()
        Dim wordsA() As String = {"cherry", "apple", "blueberry"}
        Dim wordsB() As String = {"apple", "blueberry", "cherry"}

        Dim match = wordsA.SequenceEqual(wordsB)

        Console.WriteLine("The sequences match: " & match)
    End Sub

    Public Class CustomSequenceOperators
        Public Shared Function Combine(Of T)( _
            ByVal first As IEnumerable(Of T), ByVal second As IEnumerable(Of T), ByVal func As Func(Of T, T, T)) As IEnumerable(Of T)

            Dim list As New List(Of T)
            Using e1 = first.GetEnumerator, e2 = second.GetEnumerator
                While (e1.MoveNext AndAlso e2.MoveNext)
                    list.Add(func(e1.Current, e2.Current))
                End While
            End Using
            Return list
        End Function
    End Class


    <Category("查询执行")> _
    <Title("延迟执行")> _
    <Description("以下示例显示查询是如何延迟到" & _
                 "在 For Each 语句中枚举该查询时才执行的。")> _
    <LinkedFunction("Increment")> _
    Public Sub Linq99()
        ' 序列运算符形成直到被枚举时
        ' 才会执行的高级查询。

        Dim numbers() As Integer = {5, 4, 1, 3, 9, 8, 6, 7, 2, 0}

        Dim i As Integer = 0
        Dim numberQuery = From num In numbers _
                          Select Increment(i)

        ' 请注意，计算每个元素后，局部变量“i”
        ' 才会递增(这是一种副作用):
        For Each number In numberQuery
            Console.WriteLine("number = " & number & ", i = " & i)
        Next
    End Sub

    Function Increment(ByRef i As Integer) As Integer
        i += 1
        Return i
    End Function

    <Category("查询执行")> _
    <Title("立即执行")> _
    <Description("以下示例显示如何才能用 ToList() 这样的运算符立即执行" & _
                 "查询。")> _
    <LinkedFunction("Increment")> _
    Public Sub Linq100()
        ' 类似 ToList() 的方法会使查询
        ' 立即执行，同时缓存结果。

        Dim numbers() As Integer = {5, 4, 1, 3, 9, 8, 6, 7, 2, 0}

        Dim i As Integer = 0
        Dim q = (From num In numbers Select Increment(i)).ToList()

        ' 在迭代以下结果之前，
        ' 局部变量 i 已递增到最大值:
        For Each v In q
            Console.WriteLine("v = " & v & ", i = " & i)
        Next
    End Sub

    <Category("查询执行")> _
    <Title("查询重用")> _
    <Description("以下示例显示由于延迟执行，如何才能在" & _
                 "数据更改后再次使用查询以及随后查询将如何对新数据产生作用。")> _
    Public Sub Linq101()
        ' 使用延迟执行可以只定义查询一次，
        ' 然后在数据更改后重复使用该查询。 

        Dim numbers() As Integer = {5, 4, 1, 3, 9, 8, 6, 7, 2, 0}
        Dim lowNumbers = From num In numbers _
                         Where num <= 3

        Console.WriteLine("First run numbers <= 3:")
        For Each n In lowNumbers
            Console.WriteLine(n)
        Next

        For i As Integer = 0 To numbers.Length - 1
            numbers(i) -= 1
        Next

        ' 第二次运行过程中，相同的查询对象
        ' lowNumbers 将循环访问
        ' numbers() 的新状态，从而产生不同的结果:
        Console.WriteLine("Second run numbers <= 3:")
        For Each n In lowNumbers
            Console.WriteLine(n)
        Next
    End Sub

    <Category("联接(Join/Group Join)")> _
    <Title("交叉联接")> _
    <Description("此示例显示如何根据两个序列上键表达式是否相等而" & _
                 "高效地联接这两个序列的元素。")> _
    Public Sub Linq102()
        Dim categories() As String = {"Beverages", "Condiments", "Vegetables", "Dairy Products", "Seafood"}

        Dim products = GetProductList()

        Dim categorizedProducts = From cat In categories _
                                  Join prod In products On cat Equals prod.Category _
                                  Select Category = cat, prod.ProductName

        For Each v In categorizedProducts
            Console.WriteLine(v.ProductName & ": " & v.Category)
        Next
    End Sub


    <Category("联接(Join/Group Join)")> _
    <Title("分组联接")> _
    <Description("使用分组联接可以获取所有符合给定类别" & _
                 "并组成一个序列的产品。")> _
    Public Sub Linq103()
        Dim categories As String() = {"Beverages", "Condiments", "Vegetables", "Dairy Products", "Seafood"}

        Dim productList = GetProductList()

        Dim categorizedProducts = From cat In categories _
                                  Group Join prod In productList On cat Equals prod.Category _
                                  Into Products = Group _
                                  Select Category = cat, Products

        For Each v In categorizedProducts
            Console.WriteLine(v.Category & ":")
            For Each p In v.Products
                Console.WriteLine("   " & p.ProductName)
            Next
        Next

    End Sub

    <Category("联接(Join/Group Join)")> _
    <Title("交叉联接与分组联接")> _
    <Description("Group Join 运算符比 Join 应用更广泛，正如交叉联接示例的" & _
                 "这个略显冗长的版本所示。")> _
    Public Sub Linq104()
        Dim categories() As String = {"Beverages", "Condiments", "Vegetables", "Dairy Products", "Seafood"}

        Dim productList = GetProductList()

        Dim categorizedProducts = From cat In categories _
                                  Group Join prod In productList On cat Equals prod.Category _
                                  Into Group _
                                  From prod2 In Group _
                                  Select Category = cat, prod2.ProductName

        For Each v In categorizedProducts
            Console.WriteLine(v.ProductName & ": " & v.Category)
        Next
    End Sub

    <Category("联接(Join/Group Join)")> _
    <Title("左外部联接")> _
    <Description("所谓的外部联接可以用分组联接表示。左外部联接" & _
                 "类似于交叉联接，但左侧的所有元素都至少被纳入一次，" & _
                 "即使这些元素与右侧的任何元素都不匹配也是如此。请注意 Vegetables" & _
                 "是如何在即使没有任何匹配产品的情况下仍出现在输出中的。")> _
    Public Sub Linq105()
        Dim categories() As String = {"Beverages", "Condiments", "Vegetables", "Dairy Products", "Seafood"}

        Dim productList = GetProductList()

        Dim joinResults = From cat In categories _
                          Group Join prod In productList On cat Equals prod.Category Into Group _
                          From prod2 In Group.DefaultIfEmpty() _
                          Select Category = cat, _
                                 ProductName = If(prod2 Is Nothing, "(None)", prod2.ProductName)

        For Each v In joinResults
            Console.WriteLine(v.Category & ": " & v.ProductName)
        Next
    End Sub


    <Category("* 示例数据 *")> _
    <Title("CustomerList / ProductList")> _
    <Description("此方法显示上面的查询所使用的示例数据。还可以看到" & _
                 "下面构造列表的方法。使用" & _
                 "对象初始值设定项生成 ProductList，而 CustomerList 使用 XLinq 将其值" & _
                 "从 XML 文档读入内存中。")> _
    <LinkedFunction("GetCustomerList")> _
    <LinkedFunction("GetProductList")> _
    <LinkedMethod("createLists")> _
    Public Sub Linq106()
        ObjectDumper.Write(GetCustomerList(), 1)

        Console.WriteLine()

        ObjectDumper.Write(GetProductList())
    End Sub

    <Category("基于任务的示例"), _
    Title("使用日期"), _
    Description("应用场景: 您需要将牙医预约排入已排得很满的日程中。" & _
                "牙医给了您一个列表，上面是这个月他可以安排与您会面的日期。" & _
                "使用以下查询确定哪些日期对双方都合适。")> _
    Public Sub Linq107()
        Randomize()

        Dim y = Year(Now)
        Dim m = Month(Now)
        Try

            '随机选取本月的 15 个预约日期
            Dim DentistDates(15) As Date, AvailableDates(8) As Date
            For i = 0 To UBound(DentistDates)
                DentistDates(i) = New Date(y, m, CInt((Date.DaysInMonth(y, m) - 1) * Rnd() + 1))
            Next

            '随机选取您在本月的 8 个空闲日期
            For i = 0 To UBound(AvailableDates)
                AvailableDates(i) = New Date(y, m, CInt((Date.DaysInMonth(y, m) - 1) * Rnd() + 1))
            Next

            Dim CommonDates = From day In AvailableDates, day2 In DentistDates _
                              Where day.Date = day2.Date _
                              Order By day _
                              Select day Distinct

            '为了更方便看到查询生效，我们对原始列表进行排序，然后再显示
            Array.Sort(DentistDates)
            Array.Sort(AvailableDates)

            Console.WriteLine("Dentist is free on:")
            ObjectDumper.Write(DentistDates)

            Console.WriteLine(vbNewLine & "You are free on:")
            ObjectDumper.Write(AvailableDates)

            Console.WriteLine(vbNewLine & "Appointment dates that work:")
            For Each d In CommonDates
                Console.WriteLine(Format(d, "MM/dd/yy"))
            Next
        Catch ex As Exception
            Stop
        End Try
    End Sub

    <Category("基于任务的示例"), _
    Title("System.Diagnostics"), _
    Description("此查询确定当前加载的使用内存最多的前 5 个应用程序。")> _
    Public Sub Linq108()

        Dim pList = From proc In System.Diagnostics.Process.GetProcesses() _
                    Select ProcessName = proc.ProcessName, _
                           Size = (Format(proc.WorkingSet64 / 1000, "#,##0") & " KB"), _
                           Size64 = proc.WorkingSet64 _
                    Order By Size64 _
                    Take 5

        Console.WriteLine("These 5 processes are using the most memory:")

        For Each p In pList
            Console.WriteLine(p.ProcessName & " - " & p.Size)
        Next

    End Sub

    <Category("基于任务的示例"), _
    Title("使用文件系统 - 1"), _
    Description("此查询查找去年内创建的文件。")> _
    Public Sub Linq110()

        Dim fileList = From file In New DirectoryInfo("C:\").GetFiles() _
                       Where file.CreationTime.AddYears(1) > Now _
                       Order By file.Name

        Console.WriteLine("Files created within the last year:")
        For Each f In fileList
            Console.WriteLine(f.Name)
        Next

    End Sub


    <Category("基于任务的示例"), _
    Title("使用文件系统 - 2"), _
    Description("此查询显示所有已映射的网络驱动器")> _
    Public Sub Linq111()

        '请注意，如果未映射任何网络驱动器，则查询将不会
        '返回任何结果

        '如果要查看其他驱动器类型(如 CDROM)，请更改下面的枚举
        Dim Drives = From drive In My.Computer.FileSystem.Drives() _
                     Where drive.DriveType = System.IO.DriveType.Network

        Console.WriteLine("Network Drives:")
        For Each d In Drives
            Console.WriteLine(d.Name)
        Next

    End Sub

    <Category("基于任务的示例"), _
    Title("命令行参数"), _
    Description("使用 LINQ 检查是否有 /help 命令行参数。")> _
    Public Sub Linq112()
        '要试验此参数，可以双击“我的项目”，单击“调试”并
        '在命令行参数文本框中输入“/help”

        Dim cmdArgs = Aggregate cmd In My.Application.CommandLineArgs() _
                      Into LongCount(cmd = "/help")

        If cmdArgs = 0 Then
            Console.WriteLine("""/help"" was not entered as a command line argument.")
        Else
            Console.WriteLine("""/help"" requested by user on the command line.")
        End If

    End Sub

    <Category("基于任务的示例"), _
    Title("使用注册表 - 1"), _
    Description("显示 HKLM\Software 下所有以字母 C 开头的项。")> _
    Public Sub Linq113()

        Dim SubKeys = My.Computer.Registry.LocalMachine.OpenSubKey("Software").GetSubKeyNames

        Dim reg = From regKey In SubKeys _
                  Where regKey.StartsWith("C") _
                  Order By regKey

        ObjectDumper.Write(reg)
    End Sub

    <Category("基于任务的示例"), _
    Title("使用注册表 - 2"), _
    Description("显示 HKLM\Software 和 HKCU\Software 所共有的项")> _
    Public Sub Linq114()

        Dim LMKeys = _
            My.Computer.Registry.LocalMachine.OpenSubKey("Software").GetSubKeyNames

        Dim UserKeys = _
            My.Computer.Registry.CurrentUser.OpenSubKey("Software").GetSubKeyNames

        '求两个数组的交集
        Dim reg = From LocalMachineKey In LMKeys, UserKey In UserKeys _
                  Where LocalMachineKey = UserKey _
                  Select LocalMachineKey, UserKey _
                  Order By LocalMachineKey

        Console.WriteLine("Keys common to HKLM\Software and HKCU\Software:")

        For Each r In reg
            Console.WriteLine(r.LocalMachineKey)
        Next
    End Sub

    <Category("基于任务的示例"), _
    Title("最近使用的文档历史记录"), _
    Description("显示“我最近的文档”下所有内容的快捷方式")> _
    Public Sub Linq115()

        Dim RecentPath As String = _
            Environment.GetFolderPath(Environment.SpecialFolder.Recent)

        '使用匿名类型减少返回的信息量
        Dim Recent = From file In New DirectoryInfo(RecentPath).GetFiles _
                     Select file.Name, file.LastAccessTime

        Console.WriteLine("Shortcuts to My Recent Documents:")

        For Each r In Recent
            Console.WriteLine(r.Name & " - " & r.LastAccessTime)
        Next
    End Sub

    <Category("基于任务的示例"), _
    Title("IE 收藏夹 - 1"), _
    Description("计算您的收藏夹文件夹中有多少项")> _
    Public Sub Linq116()

        Dim FavPath As String = Environment.GetFolderPath(Environment.SpecialFolder.Favorites)

        Dim FavSize = Aggregate file In New DirectoryInfo(FavPath).GetFiles() _
                      Into Count()

        Console.WriteLine("There are " & FavSize & " files in your Favorites folder.")
    End Sub

    <Category("基于任务的示例"), _
    Title("反射 - 1"), _
    Description("使用反射显示当前加载的所有系统程序集")> _
    Public Sub Linq117()

        Dim Assemblies = From assembly In My.Application.Info.LoadedAssemblies _
                         Where assembly.GetName().Name.StartsWith("System.") _
                         Select assembly.GetName().Name

        Console.WriteLine("Currently-loaded System assemblies:")
        For Each a In Assemblies
            Console.WriteLine(a)
        Next
    End Sub

    <Category("基于任务的示例"), _
    Title("反射 - 2"), _
    Description("显示此程序集中的所有公共方法。请注意有些方法是重复的。" & _
                "请参阅“反射 - 3”查看 Distinct 列表。")> _
    Public Sub Linq118()

        Dim MethodList = From type In Assembly.GetExecutingAssembly.GetTypes(), _
                              method In type.GetMembers() _
                         Where method.MemberType = MemberTypes.Method _
                               AndAlso CType(method, MethodInfo).IsPublic _
                         Select Item = CType(method, MethodInfo) _
                         Order By Item.Name

        For Each m In MethodList
            Console.WriteLine(m.Name)
        Next
    End Sub

    <Category("基于任务的示例"), _
    Title("反射 - 3"), _
    Description("显示此程序集中的所有公共方法，已移除重复的方法。" & _
                "请注意使用了嵌套的 Select。")> _
    Public Sub Linq119()

        Dim NameList = From method In _
                           (From type In Assembly.GetExecutingAssembly.GetTypes(), _
                                 method2 In type.GetMembers() _
                            Where method2.MemberType = MemberTypes.Method AndAlso _
                                  CType(method2, MethodInfo).IsPublic _
                            Select Item = CType(method2, MethodInfo) _
                            Order By Item.Name) _
                       Select method.Name _
                       Distinct

        For Each m In NameList
            Console.WriteLine(m)
        Next

    End Sub

    Public Function GetProductList() As List(Of Product)
        If productList Is Nothing Then
            CreateLists()
        End If

        Return productList
    End Function

    Public Function GetCustomerList() As List(Of Customer)
        If customerList Is Nothing Then
            CreateLists()
        End If

        Return customerList
    End Function

    Private Function MakeList(Of T)(ByVal ParamArray list() As T) As IEnumerable(Of T)
        Return list
    End Function

    Private Sub CreateLists()
        ' 在内存中创建产品数据。
        ' 使用上面的 MakeList 方法用一行代码就能达到这一效果
        productList = New List(Of Product)
        productList.Add(New Product With {.ProductID = 1, .ProductName = "Chai", .Category = "Beverages", .UnitPrice = 18D, .UnitsInStock = 39})
        productList.Add(New Product With {.ProductID = 2, .ProductName = "Chang", .Category = "Beverages", .UnitPrice = 19D, .UnitsInStock = 17})
        productList.Add(New Product With {.ProductID = 3, .ProductName = "Aniseed Syrup", .Category = "Condiments", .UnitPrice = 10D, .UnitsInStock = 13})
        productList.Add(New Product With {.ProductID = 4, .ProductName = "Chef Anton's Cajun Seasoning", .Category = "Condiments", .UnitPrice = 22D, .UnitsInStock = 53})
        productList.Add(New Product With {.ProductID = 5, .ProductName = "Chef Anton's Gumbo Mix", .Category = "Condiments", .UnitPrice = 21.35D, .UnitsInStock = 0})
        productList.Add(New Product With {.ProductID = 6, .ProductName = "Grandma's Boysenberry Spread", .Category = "Condiments", .UnitPrice = 25D, .UnitsInStock = 120})
        productList.Add(New Product With {.ProductID = 7, .ProductName = "Uncle Bob's Organic Dried Pears", .Category = "Produce", .UnitPrice = 30D, .UnitsInStock = 15})
        productList.Add(New Product With {.ProductID = 8, .ProductName = "Northwoods Cranberry Sauce", .Category = "Condiments", .UnitPrice = 40D, .UnitsInStock = 6})
        productList.Add(New Product With {.ProductID = 9, .ProductName = "Mishi Kobe Niku", .Category = "Meat/Poultry", .UnitPrice = 97D, .UnitsInStock = 29})
        productList.Add(New Product With {.ProductID = 10, .ProductName = "Ikura", .Category = "Seafood", .UnitPrice = 31D, .UnitsInStock = 31})
        productList.Add(New Product With {.ProductID = 11, .ProductName = "Queso Cabrales", .Category = "Dairy Products", .UnitPrice = 21D, .UnitsInStock = 22})
        productList.Add(New Product With {.ProductID = 12, .ProductName = "Queso Manchego La Pastora", .Category = "Dairy Products", .UnitPrice = 38D, .UnitsInStock = 86})
        productList.Add(New Product With {.ProductID = 13, .ProductName = "Konbu", .Category = "Seafood", .UnitPrice = 6D, .UnitsInStock = 24})
        productList.Add(New Product With {.ProductID = 14, .ProductName = "Tofu", .Category = "Produce", .UnitPrice = 23.25D, .UnitsInStock = 35})
        productList.Add(New Product With {.ProductID = 15, .ProductName = "Genen Shouyu", .Category = "Condiments", .UnitPrice = 15.5D, .UnitsInStock = 39})
        productList.Add(New Product With {.ProductID = 16, .ProductName = "Pavlova", .Category = "Confections", .UnitPrice = 17.45D, .UnitsInStock = 29})
        productList.Add(New Product With {.ProductID = 17, .ProductName = "Alice Mutton", .Category = "Meat/Poultry", .UnitPrice = 39D, .UnitsInStock = 0})
        productList.Add(New Product With {.ProductID = 18, .ProductName = "Carnarvon Tigers", .Category = "Seafood", .UnitPrice = 62.5D, .UnitsInStock = 42})
        productList.Add(New Product With {.ProductID = 19, .ProductName = "Teatime Chocolate Biscuits", .Category = "Confections", .UnitPrice = 9.2D, .UnitsInStock = 25})
        productList.Add(New Product With {.ProductID = 20, .ProductName = "Sir Rodney's Marmalade", .Category = "Confections", .UnitPrice = 81D, .UnitsInStock = 40})
        productList.Add(New Product With {.ProductID = 21, .ProductName = "Sir Rodney's Scones", .Category = "Confections", .UnitPrice = 10D, .UnitsInStock = 3})
        productList.Add(New Product With {.ProductID = 22, .ProductName = "Gustaf's Knäckebröd", .Category = "Grains/Cereals", .UnitPrice = 21D, .UnitsInStock = 104})
        productList.Add(New Product With {.ProductID = 23, .ProductName = "Tunnbröd", .Category = "Grains/Cereals", .UnitPrice = 9D, .UnitsInStock = 61})
        productList.Add(New Product With {.ProductID = 24, .ProductName = "Guaraná Fantástica", .Category = "Beverages", .UnitPrice = 4.5D, .UnitsInStock = 20})
        productList.Add(New Product With {.ProductID = 25, .ProductName = "NuNuCa Nuß-Nougat-Creme", .Category = "Confections", .UnitPrice = 14D, .UnitsInStock = 76})
        productList.Add(New Product With {.ProductID = 26, .ProductName = "Gumbär Gummibärchen", .Category = "Confections", .UnitPrice = 31.23D, .UnitsInStock = 15})
        productList.Add(New Product With {.ProductID = 27, .ProductName = "Schoggi Schokolade", .Category = "Confections", .UnitPrice = 43.9D, .UnitsInStock = 49})
        productList.Add(New Product With {.ProductID = 28, .ProductName = "Rössle Sauerkraut", .Category = "Produce", .UnitPrice = 45.6D, .UnitsInStock = 26})
        productList.Add(New Product With {.ProductID = 29, .ProductName = "Thüringer Rostbratwurst", .Category = "Meat/Poultry", .UnitPrice = 123.79D, .UnitsInStock = 0})
        productList.Add(New Product With {.ProductID = 30, .ProductName = "Nord-Ost Matjeshering", .Category = "Seafood", .UnitPrice = 25.89D, .UnitsInStock = 10})
        productList.Add(New Product With {.ProductID = 31, .ProductName = "Gorgonzola Telino", .Category = "Dairy Products", .UnitPrice = 12.5D, .UnitsInStock = 0})
        productList.Add(New Product With {.ProductID = 32, .ProductName = "Mascarpone Fabioli", .Category = "Dairy Products", .UnitPrice = 32D, .UnitsInStock = 9})
        productList.Add(New Product With {.ProductID = 33, .ProductName = "Geitost", .Category = "Dairy Products", .UnitPrice = 2.5D, .UnitsInStock = 112})
        productList.Add(New Product With {.ProductID = 34, .ProductName = "Sasquatch Ale", .Category = "Beverages", .UnitPrice = 14D, .UnitsInStock = 111})
        productList.Add(New Product With {.ProductID = 35, .ProductName = "Steeleye Stout", .Category = "Beverages", .UnitPrice = 18D, .UnitsInStock = 20})
        productList.Add(New Product With {.ProductID = 36, .ProductName = "Inlagd Sill", .Category = "Seafood", .UnitPrice = 19D, .UnitsInStock = 112})
        productList.Add(New Product With {.ProductID = 37, .ProductName = "Gravad lax", .Category = "Seafood", .UnitPrice = 26D, .UnitsInStock = 11})
        productList.Add(New Product With {.ProductID = 38, .ProductName = "Côte de Blaye", .Category = "Beverages", .UnitPrice = 263.5D, .UnitsInStock = 17})
        productList.Add(New Product With {.ProductID = 39, .ProductName = "Chartreuse verte", .Category = "Beverages", .UnitPrice = 18D, .UnitsInStock = 69})
        productList.Add(New Product With {.ProductID = 40, .ProductName = "Boston Crab Meat", .Category = "Seafood", .UnitPrice = 18.4D, .UnitsInStock = 123})
        productList.Add(New Product With {.ProductID = 41, .ProductName = "Jack's New England Clam Chowder", .Category = "Seafood", .UnitPrice = 9.65D, .UnitsInStock = 85})
        productList.Add(New Product With {.ProductID = 42, .ProductName = "Singaporean Hokkien Fried Mee", .Category = "Grains/Cereals", .UnitPrice = 14D, .UnitsInStock = 26})
        productList.Add(New Product With {.ProductID = 43, .ProductName = "Ipoh Coffee", .Category = "Beverages", .UnitPrice = 46D, .UnitsInStock = 17})
        productList.Add(New Product With {.ProductID = 44, .ProductName = "Gula Malacca", .Category = "Condiments", .UnitPrice = 19.45D, .UnitsInStock = 27})
        productList.Add(New Product With {.ProductID = 45, .ProductName = "Rogede sild", .Category = "Seafood", .UnitPrice = 9.5D, .UnitsInStock = 5})
        productList.Add(New Product With {.ProductID = 46, .ProductName = "Spegesild", .Category = "Seafood", .UnitPrice = 12D, .UnitsInStock = 95})
        productList.Add(New Product With {.ProductID = 47, .ProductName = "Zaanse koeken", .Category = "Confections", .UnitPrice = 9.5D, .UnitsInStock = 36})
        productList.Add(New Product With {.ProductID = 48, .ProductName = "Chocolade", .Category = "Confections", .UnitPrice = 12.75D, .UnitsInStock = 15})
        productList.Add(New Product With {.ProductID = 49, .ProductName = "Maxilaku", .Category = "Confections", .UnitPrice = 20D, .UnitsInStock = 10})
        productList.Add(New Product With {.ProductID = 50, .ProductName = "Valkoinen suklaa", .Category = "Confections", .UnitPrice = 16.25D, .UnitsInStock = 65})
        productList.Add(New Product With {.ProductID = 51, .ProductName = "Manjimup Dried Apples", .Category = "Produce", .UnitPrice = 53D, .UnitsInStock = 20})
        productList.Add(New Product With {.ProductID = 52, .ProductName = "Filo Mix", .Category = "Grains/Cereals", .UnitPrice = 7D, .UnitsInStock = 38})
        productList.Add(New Product With {.ProductID = 53, .ProductName = "Perth Pasties", .Category = "Meat/Poultry", .UnitPrice = 32.8D, .UnitsInStock = 0})
        productList.Add(New Product With {.ProductID = 54, .ProductName = "Tourtière", .Category = "Meat/Poultry", .UnitPrice = 7.45D, .UnitsInStock = 21})
        productList.Add(New Product With {.ProductID = 55, .ProductName = "Pâté chinois", .Category = "Meat/Poultry", .UnitPrice = 24D, .UnitsInStock = 115})
        productList.Add(New Product With {.ProductID = 56, .ProductName = "Gnocchi di nonna Alice", .Category = "Grains/Cereals", .UnitPrice = 38D, .UnitsInStock = 21})
        productList.Add(New Product With {.ProductID = 57, .ProductName = "Ravioli Angelo", .Category = "Grains/Cereals", .UnitPrice = 19.5D, .UnitsInStock = 36})
        productList.Add(New Product With {.ProductID = 58, .ProductName = "Escargots de Bourgogne", .Category = "Seafood", .UnitPrice = 13.25D, .UnitsInStock = 62})
        productList.Add(New Product With {.ProductID = 59, .ProductName = "Raclette Courdavault", .Category = "Dairy Products", .UnitPrice = 55D, .UnitsInStock = 79})
        productList.Add(New Product With {.ProductID = 60, .ProductName = "Camembert Pierrot", .Category = "Dairy Products", .UnitPrice = 34D, .UnitsInStock = 19})
        productList.Add(New Product With {.ProductID = 61, .ProductName = "Sirop d'érable", .Category = "Condiments", .UnitPrice = 28.5D, .UnitsInStock = 113})
        productList.Add(New Product With {.ProductID = 62, .ProductName = "Tarte au sucre", .Category = "Confections", .UnitPrice = 49.3D, .UnitsInStock = 17})
        productList.Add(New Product With {.ProductID = 63, .ProductName = "Vegie-spread", .Category = "Condiments", .UnitPrice = 43.9D, .UnitsInStock = 24})
        productList.Add(New Product With {.ProductID = 64, .ProductName = "Wimmers gute Semmelknödel", .Category = "Grains/Cereals", .UnitPrice = 33.25D, .UnitsInStock = 22})
        productList.Add(New Product With {.ProductID = 65, .ProductName = "Louisiana Fiery Hot Pepper Sauce", .Category = "Condiments", .UnitPrice = 21.05D, .UnitsInStock = 76})
        productList.Add(New Product With {.ProductID = 66, .ProductName = "Louisiana Hot Spiced Okra", .Category = "Condiments", .UnitPrice = 17D, .UnitsInStock = 4})
        productList.Add(New Product With {.ProductID = 67, .ProductName = "Laughing Lumberjack Lager", .Category = "Beverages", .UnitPrice = 14D, .UnitsInStock = 52})
        productList.Add(New Product With {.ProductID = 68, .ProductName = "Scottish Longbreads", .Category = "Confections", .UnitPrice = 12.5D, .UnitsInStock = 6})
        productList.Add(New Product With {.ProductID = 69, .ProductName = "Gudbrandsdalsost", .Category = "Dairy Products", .UnitPrice = 36D, .UnitsInStock = 26})
        productList.Add(New Product With {.ProductID = 70, .ProductName = "Outback Lager", .Category = "Beverages", .UnitPrice = 15D, .UnitsInStock = 15})
        productList.Add(New Product With {.ProductID = 71, .ProductName = "Flotemysost", .Category = "Dairy Products", .UnitPrice = 21.5D, .UnitsInStock = 26})
        productList.Add(New Product With {.ProductID = 72, .ProductName = "Mozzarella di Giovanni", .Category = "Dairy Products", .UnitPrice = 34.8D, .UnitsInStock = 14})
        productList.Add(New Product With {.ProductID = 73, .ProductName = "Röd Kaviar", .Category = "Seafood", .UnitPrice = 15D, .UnitsInStock = 101})
        productList.Add(New Product With {.ProductID = 74, .ProductName = "Longlife Tofu", .Category = "Produce", .UnitPrice = 10D, .UnitsInStock = 4})
        productList.Add(New Product With {.ProductID = 75, .ProductName = "Rhönbräu Klosterbier", .Category = "Beverages", .UnitPrice = 7.75D, .UnitsInStock = 125})
        productList.Add(New Product With {.ProductID = 76, .ProductName = "Lakkalikööri", .Category = "Beverages", .UnitPrice = 18D, .UnitsInStock = 57})
        productList.Add(New Product With {.ProductID = 77, .ProductName = "Original Frankfurter grüne Soße", .Category = "Condiments", .UnitPrice = 13D, .UnitsInStock = 32})

        ' 使用 XLinq 将客户/订单数据从 XML 文件中读入内存:
        Dim customerListPath = Path.GetFullPath(Path.Combine(dataPath, "customers.xml"))

        Dim list = XDocument.Load(customerListPath).Root.Elements("customer")

        customerList = (From e In list _
                        Select New Customer With { _
                            .CustomerID = CStr(e.<id>.Value), _
                            .CompanyName = CStr(e.<name>.Value), _
                            .Address = CStr(e.<address>.Value), _
                            .City = CStr(e.<city>.Value), _
                            .Region = CStr(e.<region>.Value), _
                            .PostalCode = CStr(e.<postalcode>.Value), _
                            .Country = CStr(e.<country>.Value), _
                            .Phone = CStr(e.<phone>.Value), _
                            .Fax = CStr(e.<fax>.Value), _
                            .Orders = ( _
                                       From o In e...<orders>...<order> _
                                       Select New Order With { _
                                            .OrderID = CInt(o.<id>.Value), _
                                            .OrderDate = CDate(o.<orderdate>.Value), _
                                            .Total = CDec(o.<total>.Value)}).ToArray() _
                        }).ToList()
    End Sub

End Class